﻿//
// Panel CCI w v2011.0 
// Copyright (C) 2011, Majestas Traders www.majestastrader.com
// Copyright (C) 2011, Brian Peay <GainsMaker>.
// Portions copyright (C) 2006, NinjaTrader LLC <ninjatrader@ninjatrader.com>.
// Portions copyright (C) 2007, Martin Stride <MartinS>
// Portions copyright (C) 2007, Scott Elder <ScottEnAZ>

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Display the CCIPanel with OHL of day prices, HL of prior and current bar, current price,
	/// prior close price, bar percentage complete, bar countdown timer, EMA slope/distance and ZLR points indicator.
	/// 
	/// Feel free to add to this code and please release a new version to the Woodies Forum. 
	/// Please release it as a beta version so that folks know what they are getting into. We can keep
	/// the beta and the latest stable version posted for download.
	/// Make sure you add your name and copyright line at the top.
	/// 
	/// Changes from v1.1 by MartinS described in following 3 lines:
	/// 34EMA distance from price and 34EMA slope added (slope compares current ema with the average of the prior 2 bars).
	/// Price High/Low will change color if it makes a higher high or lower low compared with its prior bar.
	/// Removed references to DataSeries for CCI, High and Low values
	/// 
	/// Changes from v1.2 by MartinS described in following 12 lines:
	/// Added user inputs to control the vertical position of several of the display items (uses % down from top)
	/// HoD/LoD/Open layout changed to make the above more manageable.
	/// Added user inputs to enable/disable display of several items (Woodie does NOT use most of these).
	/// Added countdown timer for Tick and Volume charts - seleced automatically depending on chart type.
	/// Added user inputs for remaining time/ticks/volume alert on countdown timer.
	/// Added new timer format of "Min:Sec" as alternative to total seconds countdown (user selectable).
	/// Added optional display of prior bar close price.
	/// Changed panel background color (my preference) - this could be a user input(?), but I don't know how to catch input errors.
	/// Added user input to use this panel color or use own chart color.
	/// Added trailing zeros to price display (just my preference).
	/// Added additional criteria for color change to the CCI points difference display -
	/// added items; CCI trend / been outside 100 line / CCI direction / correct side of zero line must all agree.  
	/// 
	/// Changes from v1.3 by MartinS described in following 4 lines:
	/// Added extra color change stage to CCI points difference - now yellow at 15(ok) then green at 20(very good).
	/// Various changes to user input layout (gets rid of input parameters from top of chart display).
	/// Changed minutes Timer ~ copy from NinjaTrader code.  Forces update if no new tick for 1 second. Works with replay.
	/// Added LSMA slope and distance from price in ticks (slope compares current lsma with the average of the prior 2 bars).
	/// 
	/// Changes from v1.4 by Scott Elder (ScottEnAZ) in following 3 lines:
	/// Added audio alert functionality to Minute, Second, Tick and Volume period charts - triggered on alertTimerLast.
	/// Added user selectable audio file to be played on timer alert (user selectable - panel parameters).
	/// Added enable/disable functionality to the audio alert functionality (user selectable - panel parameters).
	/// 
	/// Changes from v1.5 by MartinS described in following 4 lines:
	/// Added "timer" support for Range Bars (NinjaTrader v6.5 onwards):-
	/// - counts down ticks left before current bar size = the range bar setting for chart
	/// - alert text color change at a user selected number of ticks before range bar size is reached (enter ticks at "AlertTimerLast")
	/// - Max possible High and Min possible Low for current bar are shown to right of "timer" (one of these will be the bar close value).
	/// 
	/// Changes from v1.6 by MartinS described in following 2 lines:
	/// Added Prior and Current bar CCI value ~ user input to control position.
	/// Add support for the new "RangeW" bar type, which does not fill gaps in price.
	/// 
	/// Changes from v1.7  by MartinS described in following 4 lines:
	/// Minutes timer should now work OK when session time spans midnight.
	/// Changes to automatic decimal places detection~ to cope with 6 decimals (Japanese Yen).
	/// Added user input for panel width, and display numbers will center themselves within width selected (allows wider panel for longer numbers).
	/// NOTE- Chart Properties "Right side margin" has to be adjusted manually to match panel width, or chart data may disappear behind the panel.
	/// </summary>
	
	[Description("The CCI panel w")] 	// the "label" shown when selected from available indicators list.
    [Gui.Design.DisplayName("CCI Panel w")]	 // name as shown in available indicators list.
    public class CCIpanel_w : Indicator
    {
        #region Variables
        // Wizard generated variables
        private int period = 14; // Default setting for Period
		private int location_Timer = 15; // Default setting for Location_Timer
		private bool showTimerPercent = false; // Default setting for ShowTimerPercent
		private bool showTimerAsSeconds = false; // Default setting for ShowTimerAsSeconds
		private int alertTimer1st = 5; // Default setting for AlertTimer1st
		private int alertTimerLast = 2; // Default setting for AlertTimerLast
        private int location_Price = 40; // Default setting for Location_Price
        private bool showLastClose = true; // Default setting for ShowLastClose
		private bool showDayHiLo = false; // Default setting for ShowDayHiLo
		private int location_DayHiLo = 75; //Default setting for Location_DayHiLo
		private bool showEMA = false; // Default setting for ShowEMA
		private int location_EMA = 60; // Default setting for Location_EMA
		private bool showLSMA = false; // Default setting for ShowLSMA
		private int location_LSMA = 65; // Default setting for Location_LSMA
		private bool showCCIpoints = true; //Default setting for ShowCCIpoints
		private int location_CCIpoints = 88; // Default setting for Location_CCIpoints
		private int location_CCIvalue = 84;	 // Default setting for Location_CCIvalue
		private bool showPanelColor = true; // Default setting for ShowPanelColor
		private int panelLeftEdge = 125;  // Default setting for PanelLeftEdge
		private int threshold_DayHiLo = 5; // Default setting for Threshold_HoD_LoD
        // User defined variables (add any user defined variables below)
		private StringFormat	stringFormat		= new StringFormat();		// by default will left align text
		private StringFormat	formatCenter		= new StringFormat();		// to be used to center text
		private StringFormat	formatRight			= new StringFormat();		// to be used to right align text
		private SolidBrush		textBrushCopywrite	= new SolidBrush(Color.Teal);		//copywrite lable
		private SolidBrush		textBrushNormal		= new SolidBrush(Color.Black);			//numerous uses
		private SolidBrush		textBrushVersion	= new SolidBrush(Color.Gainsboro);		//version lable
		private SolidBrush		textBrushBackground	= new SolidBrush(Color.FromArgb(113,142,120));	//background 
		private SolidBrush		textBrushHighs		= new SolidBrush(Color.Green);	//bar new highs
		private SolidBrush		textBrushLows		= new SolidBrush(Color.DarkRed);		//bar new lows
		private SolidBrush		textBrushOpen		= new SolidBrush(Color.Orange);			//price near to Day Open
		private SolidBrush		textBrushRange		= new SolidBrush(Color.Gainsboro);		//bar size(not used)
		private SolidBrush		textBrushClosePrior	= new SolidBrush(Color.Black);			//prior bar close
		private SolidBrush		textBrushTimer0		= new SolidBrush(Color.Green);	//timer seconds - final color
		private SolidBrush		textBrushTimer1		= new SolidBrush(Color.Orange);			//timer seconds - middle color
		private SolidBrush		textBrushTimer		= new SolidBrush(Color.Black);			//holder for current color
		private SolidBrush		textBrushCCIvg		= new SolidBrush(Color.Green);	//CCI points(Very Good)
		private SolidBrush		textBrushCCIok		= new SolidBrush(Color.Orange);			//CCI points(OK)
		private SolidBrush		textBrushCCInotOK	= new SolidBrush(Color.Red);			//CCI points(not OK)
		private SolidBrush		textBrushCCIprior	= new SolidBrush(Color.Black);		//CCI points prior bar
		private SolidBrush		textBrushLime		= new SolidBrush(Color.Lime);			//in testing section only

		// private Pen				penBlack			= new Pen(Color.Black);					//left edge of panel
		private Font		textFontLabels		= new Font("Bookman Old Style", 7);					//copywrite+version lables
		private Font		textFontSmValues	= new Font("Bookman Old Style", 9);					//most text
		private Font		textFontValues		= new Font("Bookman Old Style", 14);					//current price+timer+CCI points
		// private float		textWidthLabels		= 0;
		// private float		textHeightLabels	= 0;
		private float		textWidthValues		= 0;
		private float		textHeightValues	= 0;
		private float		textWidthSmValues	= 0;
		// private float		textHeightSmValues	= 0;
		// private int			vSize				= 0;
		private DataSeries	cciData;				 // need CCI values available for ZLR detection (part of CCI diff code)
		private int 		cciPointsDifference	= 0;
		private int 		cciPointsPriorDiff	= 0;
		private double		closeCurrentBar		= 0;
		private double		closePriorBar		= 0;
		private double		highPrior2Bar		= 0; //this is the bar before "PriorBar" (for text color change detection)
		private double		lowPrior2Bar		= 0;
		private double		highPriorBar		= 0;
		private double		lowPriorBar			= 0;
		private double		highCurrentBar		= 0;
		private double		lowCurrentBar		= 0;
		private double		openOfDay			= 0;
		private double		highOfDay			= 0;
		private double		lowOfDay			= 0;
		private int			openDistance		= 0;
		private int			hodDistance			= 0;
		private int			lodDistance			= 0;
		private int			currentPercent		= 0; // for % complete display
		// private	int			remainingPercent	= 0;
		private System.Windows.Forms.Timer		timer;	// for minutes timer
		private string		timeLeft			= "timer";
		private int			remainingTicks		= 0;
		private int			remainingVol		= 0;
		private int			ema34distance		= 0;
		private int			ema34slope			= 0;
		private int			lsma25distance		= 0;
		private int			lsma25slope			= 0;
		private double		radToDegrees		= 180/Math.PI; // to convert Radians to Degrees for slope calc
		private int			zlrMax				= 120;// 120 MAX on the current CCI bar is the entry limit.
		private int			zlrMin				= 15; // 15 CCI points is the accepted nuance right now. This might be made user adjustable in the future.
		private double		cciValue			= 0;  // current CCI value ~ for print in panel
		private double		cciValuePrior		= 0;  // prior bar CCI value ~ for print in panel
		private int			countZeroX			= 0;	// track bars ago last cross above/below Zero line
		private int			count100L			= 99;	// track bars ago last outside 100
		private int			count100S			= -99;
		private int			trendCCI			= 0;  // Up+1 Down-1 (potential for NoTrend=0)
		private int			trendCCIpriorBar	= 0;  // as above but for the prior bar
		private bool		hit100Long			= false; // status for "been outside 100 line since this trend started"
		private bool		hit100Short			= false;
		private bool		zlrOK				= false; // ZLR go/no-go flag
		private bool		zlrVG				= false; // ZLR "very good" go flag
		private string		decimals			= ""; // use to set number of decimal places for display of prices
		private bool		audioTimerAlertPlay = false;  // use to enable/disable Audio Timer Alert
		private string		audioTimerAlertFile = "Alert2.wav";  // Use to set the .wav file to be played for Audio Timer Alert
		private int			range 				= 0;	// when using range bars, this is the range setting
		private double		rangeHi				= 0;	// high value for max size range bar (based on current low)
		private double		rangeLo				= 0;	// low value for max size range bar (based on current high)
		private int			currentRange		= 0;	// tick range of current bar as it forms
		private int			remainingRange		= 0;	// ticks until range bar is at max size
		#endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
			AutoScale			= false;
            CalculateOnBarClose	= false;
            Overlay				= true;
            PriceTypeSupported	= false;
			DisplayInDataBox	= false;
			PaintPriceMarkers	= false;
			PlotsConfigurable	= true;
			Name				= "Panel CCI w";	// name as shown on chart
			
			cciData = new DataSeries(this);
        }
		
		/// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			//**************************************
			// This secion is performed on First bar only
			// Contains Set-ups that remain constant for chart, so don't need to be repeated every bar/tick
			//
			if (CurrentBar == 0)
			{
			// Set the stringformat parameter for decimal places to suit the instrument.
			// This ensures that trailing zeros are always displayed - maybe an easier way to do this?
				int tickLength;
				double temp1 = TickSize - (int)TickSize;// get fractional part of TickSize
				if (temp1 < 0.001)	// c# will use scientific notation for small numbers. Attempt to cope with this here.
				{
					decimals = (temp1*1000).ToString();	// make it into a string, after removing 3 leading zeros
					tickLength = (decimals.Length)+3;	// get the number of characters in the string(then add back the 3 previously removed)
				}
				else				// no need to cope with scientific notation(hopefully)
				{
					decimals = temp1.ToString();		// make it into a string
					tickLength = decimals.Length;		// get the number of characters in the string
				}
				switch (tickLength)						// set format depending on length of string
				{
					case 1:  decimals = "f0";	// eg  YM 1t = 1 point
						break;
					case 3:  decimals = "f1";	// eg ER2 1t = 0.1 point 
						break;
					case 4:  decimals = "f2";	// eg  NQ 1t = 0.25 point
						break;
					case 5:  decimals = "f3";	// eg 1t = 0.001 point
						break;
					case 6:  decimals = "f4";	// eg  6E 1t = 0.0001 point
						break;
					case 7:  decimals = "f5";	// eg 1t = 0.00001 point
						break;
					case 8:  decimals = "f6";	// eg 6J 1t = 0.000001 point
						break;
					default: decimals = "";		// default is no fixed decimal format
						break;
					// more options can be added if neccessary......
				}
				Print(Time[0].ToString() + "  decimals= " + decimals);
					
			// StringFormat set up
				formatCenter.Alignment = StringAlignment.Center;	// text centered
				formatRight.Alignment = StringAlignment.Far;		// text right aligned
				
			// For Range Bars only ~ get the range setting (in ticks), also use later as flag (>0=we are using range bars)
			// (".Final0" & ".Custom1" are to allow custom bar types i.e RangeW bar type)
				if (Bars.Period.Id == PeriodType.Range || Bars.Period.Id == PeriodType.Custom1 || Bars.Period.Id == PeriodType.Final1 || Bars.Period.Id == PeriodType.Final0)
					range = Bars.Period.Value;

			}
			// End of Set-up
			//**************************************

			
			//**************************************
			// This secion is performed every tick update after "Period" number of bars are present
			// If currentBar is < Period then insufficient data on chart for accurate CCI calculation.
			// Also ensures we have data to look back at for current/prior comparisons (if we try to access the DataSeries beyond it's range the code will fail).
			
			if (CurrentBar < Period) return;
			
			
			////////// BAR PRICES
			// Load the data for the High/Low/Close of price bars.
			if (FirstTickOfBar)
			{
				highPrior2Bar	= High[2];	// High of 2 bars ago
				lowPrior2Bar	= Low[2];
				highPriorBar	= High[1];	// High of 1 bar ago
				lowPriorBar		= Low[1];
				closePriorBar	= Close[1];	// Close of prior bar
			}
			highCurrentBar		= High[0];	// High of current bar
			lowCurrentBar		= Low[0];
			closeCurrentBar		= Close[0];	// Price right now
			
			
			////////// HOD LOD OPEN PRICES
			// Load current day Open/High/Low values (user input to enable)
			// and calculate distance to current price in Ticks
			if (showDayHiLo)
			{
				openOfDay	= CurrentDayOHL().CurrentOpen[0];
				highOfDay	= CurrentDayOHL().CurrentHigh[0];
				lowOfDay	= CurrentDayOHL().CurrentLow[0];
			// distance in ticks that price is above or below(-ve) the days Open price
				openDistance = Convert.ToInt32((Close[0]-openOfDay)/TickSize);
			// distance in ticks between price and HoD and LoD (always +ve)
				hodDistance	 = Convert.ToInt32((highOfDay-Close[0])/TickSize);
				lodDistance	 = Convert.ToInt32((Close[0]-lowOfDay)/TickSize);
			}
			
			////////// EMA
			// Calculate EMA angle and distance to current price (user input to enable)
			if (showEMA)
			{
			// slope compares current EMA with the average of the 2 prior bars (or the EMA 1.5 bars back)
				ema34slope = (int)(radToDegrees*(Math.Atan((EMA(34)[0]-(EMA(34)[1]+EMA(34)[2])/2)/1.5/TickSize)));
			// distance in ticks that price is above or below(-ve) the EMA line
			//ema34distance = (int)(Math.Round((Close[0]-EMA(34)[0])/TickSize,0));
				ema34distance = Convert.ToInt32((Close[0]-EMA(34)[0])/TickSize); // Convert.ToInt is efficient rounding otherwise eg ema34distance=4.99999
			}

			////////// LSMA
			// Calculate LSMA angle and distance to current price (user input to enable)
			if (showLSMA)
			{
			// slope compares current LSMA with the average of the 2 prior bars (or the LSMA 1.5 bars back)
				lsma25slope = (int)(radToDegrees*(Math.Atan((LinReg(25)[0]-(LinReg(25)[1]+LinReg(25)[2])/2)/1.5/TickSize)));
			// distance in ticks that price is above or below(-ve) the LSMA line
				lsma25distance = Convert.ToInt32((Close[0]-LinReg(25)[0])/TickSize); // Convert.ToInt is efficient rounding otherwise eg ema34distance=4.99999
			}			
			
			////////// COUNTDOWN TIMERS
			///
			/// Minutes timer code from NinjaTrader ~ "BarTimer"
			if (Bars.Period.Id == PeriodType.Minute || Bars.Period.Id == PeriodType.Second)
			{
				if (timer == null && DisplayTime())
            	{
                	timer = new System.Windows.Forms.Timer();
                	timer.Interval = 1000;
                	timer.Tick += new EventHandler(OnTimerTick);
                	timer.Enabled = true;
            	}
			}
			// For a RANGE BAR chart calc max High and min Low based on current Hi/lo, and calc current bar range
			else
			if (range > 0)
			{
				rangeHi = Low[0] + range*TickSize;
				rangeLo = High[0] - range*TickSize;
				currentRange = Convert.ToInt32((High[0] - Low[0])/TickSize);
				remainingRange = range - currentRange;
			}
			// For a TICK chart get the number of ticks remaining this bar
			else
			if (Bars.Period.Id == PeriodType.Tick)
				remainingTicks = Bars.Period.Value - Bars.TickCount;				

			// For a VOLUME chart get the volume remaining this bar
			else
			if (Bars.Period.Id == PeriodType.Volume)
				remainingVol = (int)(Bars.Period.Value - Volume[0]);
			
				
			////////// CCI DIFFERENCE BETWEEN BARS & CRITERIA FOR ZLR (user input to enable).
			// Set CCI dataseries value and Calculate CCI difference values for display.
			if (showCCIpoints)
			{
				cciData.Set(CCI(period)[0]);
				cciPointsDifference = Math.Abs((int)(cciData[0] - cciData[1]));	// CCI difference between this bar to 1 bar ago
				cciValue = cciData[0];	// current CCI value to print in panel
				if (FirstTickOfBar)
				{
					cciPointsPriorDiff = Math.Abs((int)(cciData[1] - cciData[2]));	// CCI difference between 1 bar and 2 bars ago
					cciValuePrior = cciData[1];	// prior bar close CCI value to print in panel
				}

			////////// DETERMINE IF OK FOR A ZLR
			//
			// Determine trend based on CCI.
				// Do following on FirstTickOfBar only.
				if (FirstTickOfBar)
				{
				// For prior bar, increment count of bars since last outside +/-100 lines.
					count100L = cciData[1] > 100 ? 1 : count100L + 1;
					count100S = cciData[1] < -100 ? -1 : count100S - 1; // NOTE~ increasing negative
				// For prior bar, increment count of consecutive bars above/below zero line.
					if (cciData[1] >= 0)
						countZeroX = countZeroX > 0 ? countZeroX+1: 1;
					else
						countZeroX = countZeroX < 0 ? countZeroX-1: -1; // NOTE~ increasing negative
				// For prior bar, look for move outside 100's.
				// When trend state is 1 or -1 (not been outside 100's), then look for outside 100.
				if (trendCCI == -1 && count100S >= countZeroX) // at -100 line since last move below zeroline ("count" are both -ve)
					trendCCI = -2;
				else
				if (trendCCI == 1 && count100L <= countZeroX)	 // at +100 line since last move above zeroline ("count" are both +ve)
					trendCCI = 2;
				// Update trendCCI for the prior bar, before current bar is modified
					trendCCIpriorBar = trendCCI;
				} //end of FToB
			// Only need to look for trend change on 6th bar, and change is always relative to prior bar trend.
				// Each tick, when 5 prior bars count in opposite direction then look for change to opposite trend state.
				if (countZeroX == -5 && trendCCIpriorBar > -1) // NOT short(-1,-2) AND 5 prior bars below zero, then look for change to short
					trendCCI = cciData[0] < 0 ? -1 : trendCCIpriorBar;
				else
				if (countZeroX == 5 && trendCCIpriorBar < 1)	 // NOT long(+1,+2) AND 5 bars above zero, then look for change to long
					trendCCI = cciData[0] > 0 ? 1 : trendCCIpriorBar;

			////////// Set condition for a zlrOK.
			// REMEMBER - a hook (or reject of zero), Sidewinder and CZI are not included here.
				zlrOK = ((cciPointsDifference >= zlrMin && Math.Abs(cciData[0]) < zlrMax) &&	// enough CCI difference and Not overstretched
						((trendCCI == 2 && Rising(cciData) && cciData[0] > 0) ||	// AND (UpTrend+hit100L + pointing up + above zero)
						(trendCCI == -2 && Falling(cciData) && cciData[0] < 0)));// OR (DownTrend+hit100S + pointing down + below zero)
				zlrVG = (zlrOK && cciPointsDifference >= 20);
			}
			
		}
		
		#region Miscellaneous
		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{
			if (Bars == null)
                return;
			
			// This Plot code method displays numeric values and messages in a seperate indicator pane.
			
			/////////////////////////////////////////
			// PANEL DISPLAY AREA
			/////////////////////////////////////////
			
			// Set left side reference value for Panel width (this is a reference on x-axis for text positioning~all text is ref this value)
			int widthX = bounds.X + bounds.Width - PanelLeftEdge+2;	// left edge of panel text
			int widthX25 = (int)(widthX + PanelLeftEdge*0.25);		// 25% across form left edge
			int widthX50 = (int)(widthX + PanelLeftEdge*0.5);		// 50% across form left edge
			int widthX75 = (int)(widthX + PanelLeftEdge*0.75);		// 75% across form left edge
			int widthX100 = (int)(widthX + PanelLeftEdge-4);		// almost 100% across form left edge (leave small space to right of text)
			
			// FILL PANEL area with a background color.
			// User input to enable color fill
			if (showPanelColor)
				graphics.FillRectangle(textBrushBackground, widthX - 2, bounds.Y, bounds.Width, bounds.Height);
			
			// BRAND PANEL with the © majestastrader.com
			graphics.DrawString("© Panel CCI", textFontLabels,textBrushCopywrite, widthX, bounds.Y + 10, stringFormat);
			
			/////////////////////////////////////////
			// PRICE DISPLAY
			//
			// High and low prices of prior bar and current bar, current tick price, close of prior bar, and bar ranges.
			// User input for text location - defines top of current price, with Hi / Lo / etc as fixed offsets.
			// PRIOR BAR HIGH (if higher than high of bar before it then color "Highs", else color black).
			graphics.DrawString(highPriorBar.ToString(decimals), textFontSmValues, highPriorBar>highPrior2Bar ? textBrushHighs : textBrushNormal, widthX25, bounds.Y + (bounds.Height*location_Price/100) - 20, formatCenter);
			// PRIOR BAR LOW (if lower than low of bar before it then color "Lows", else color black)
			graphics.DrawString(lowPriorBar.ToString(decimals), textFontSmValues, lowPriorBar<lowPrior2Bar ? textBrushLows : textBrushNormal, widthX25, bounds.Y + (bounds.Height*location_Price/100) + 32, formatCenter);
			// CURRENT BAR HIGH (if higher than high of bar before it then color "Highs", else color black)
			graphics.DrawString(highCurrentBar.ToString(decimals), textFontSmValues, highCurrentBar>highPriorBar ? textBrushHighs : textBrushNormal, widthX75, bounds.Y + (bounds.Height*location_Price/100) - 20, formatCenter);
			// CURRENT BAR LOW (if lower than low of bar before it then color "Lows", else color black)
			graphics.DrawString(lowCurrentBar.ToString(decimals), textFontSmValues, lowCurrentBar<lowPriorBar ? textBrushLows : textBrushNormal, widthX75, bounds.Y + (bounds.Height*location_Price/100) + 32, formatCenter);
			// CURRENT PRICE tick
			graphics.DrawString(closeCurrentBar.ToString(decimals), textFontValues,textBrushNormal, widthX50, bounds.Y + (bounds.Height*location_Price/100), formatCenter);
			// PRIOR BAR CLOSE price with user input to enable display
			if (ShowLastClose)
				graphics.DrawString(closePriorBar.ToString(decimals), textFontSmValues,textBrushClosePrior, widthX25, bounds.Y + (bounds.Height*location_Price/100) - 44, formatCenter);

			
			/////////////////////////////////////////
			// HOD/OPEN/LOD and distance to current price in ticks
			// User input to enable display and set vertical location.
			if (showDayHiLo)
			{
				// HoD and price distance in ticks. If outside threshold (user input) then color HoD black else color "Highs"
				graphics.DrawString("H: "+highOfDay.ToString(decimals), textFontSmValues, (hodDistance > threshold_DayHiLo) ? textBrushNormal : textBrushHighs, widthX+3, bounds.Y + (bounds.Height*location_DayHiLo/100)-17, stringFormat);
				graphics.DrawString(hodDistance.ToString()+"t", textFontSmValues,textBrushNormal, widthX100, bounds.Y + (bounds.Height*location_DayHiLo/100)-17, formatRight);
				// Open and price distance in ticks. If outside threshold (user input) then color Open black else color "Open"
				if (openDistance > threshold_DayHiLo || openDistance < -threshold_DayHiLo)
					graphics.DrawString("O: "+openOfDay.ToString(decimals), textFontSmValues,textBrushNormal, widthX+2, bounds.Y + (bounds.Height*location_DayHiLo/100), stringFormat);
				else
					graphics.DrawString("O: "+openOfDay.ToString(decimals), textFontSmValues,textBrushOpen, widthX+2, bounds.Y + (bounds.Height*location_DayHiLo/100), stringFormat);
				graphics.DrawString(openDistance.ToString()+"t", textFontSmValues,textBrushNormal, widthX100, bounds.Y + (bounds.Height*location_DayHiLo/100), formatRight);
				// LoD and price distance in ticks. If outside threshold (user input) then color LoD black else color "Lows"
				graphics.DrawString("L: "+lowOfDay.ToString(decimals), textFontSmValues, (lodDistance > threshold_DayHiLo) ? textBrushNormal : textBrushLows, widthX+5, bounds.Y + (bounds.Height*location_DayHiLo/100)+17, stringFormat);
				graphics.DrawString(lodDistance.ToString()+"t", textFontSmValues,textBrushNormal, widthX100, bounds.Y + (bounds.Height*location_DayHiLo/100)+17, formatRight);
			}
			
			/////////////////////////////////////////
			// 34 EMA
			// User input to enable display and set vertical location.
			if (showEMA)
			{
				graphics.DrawString("EMA: "+ema34slope.ToString()+"°", textFontSmValues,textBrushNormal, widthX,  bounds.Y + (bounds.Height*location_EMA/100), stringFormat);
				graphics.DrawString(ema34distance.ToString()+"t", textFontSmValues,textBrushNormal, widthX100,  bounds.Y + (bounds.Height*location_EMA/100), formatRight);
			}

			/////////////////////////////////////////
			// 25 LSMA
			// User input to enable display and set vertical location.
			if (showLSMA)
			{
				graphics.DrawString("lsma: "+lsma25slope.ToString()+"°", textFontSmValues,textBrushNormal, widthX,  bounds.Y + (bounds.Height*location_LSMA/100), stringFormat);
				graphics.DrawString(lsma25distance.ToString()+"t", textFontSmValues,textBrushNormal, widthX100,  bounds.Y + (bounds.Height*location_LSMA/100), formatRight);
			}
			
			/////////////////////////////////////////
			// COUNTDOWN TIMER & TICKER
			//
			// User input for text location - defines top of Tick/Volume/Seconds Remaining, with % Complete as fixed offset.
			// For Time chart less than daily interval (intra-day).
			if (Bars.Period.Id == PeriodType.Minute || Bars.Period.Id == PeriodType.Second)
			{
				if (DisplayTime())
				{
					if (timer != null && !timer.Enabled)
						timer.Enabled = true;
					TimeSpan barTimeLeft = Bars.Get(Bars.Count - 1).Time.Subtract(Now.AddSeconds(-0.25));	// -0.25 to allow for factional parts of seconds
				// Set timer text color.
					if (barTimeLeft <= new TimeSpan(00, 00, alertTimerLast)){	// within last timer alert color period
						textBrushTimer = textBrushTimer0;						// set color accordingly
						// If audio alert flag is on, sound alert and rearm after counter hits zero
						if(audioTimerAlertPlay && barTimeLeft.Seconds >0)
							SoundTimerAlert(barTimeLeft.Seconds+1);
					}
					else
					if (barTimeLeft <= new TimeSpan(00, 00, alertTimer1st))		// within first timer alert color period
						textBrushTimer = textBrushTimer1;						// set color accordingly
					else
						textBrushTimer = textBrushNormal;						// otherwise color = normal
				// Set format for timer display.
				// Set "seconds only" format.
					if (showTimerAsSeconds)
					{
						timeLeft = barTimeLeft.TotalSeconds.ToString("0") + "s";
						graphics.DrawString(timeLeft, textFontValues, textBrushTimer, widthX75, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
					}
				// Set "Hrs:Min:Sec" format so that we don't have any leading "00"
					else
					if (barTimeLeft < new TimeSpan(00, 01, 00))	// < 1 minute, then show seconds only
					{
						timeLeft = barTimeLeft.Seconds.ToString("0") + "s";
						graphics.DrawString(timeLeft, textFontValues, textBrushTimer, widthX75, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
					}
					else
					if (barTimeLeft < new TimeSpan(01, 00, 00))	// < 1 hour, then show min:sec format
					{
						timeLeft = barTimeLeft.Minutes.ToString("00") + ":" + barTimeLeft.Seconds.ToString("00");
						graphics.DrawString(timeLeft, textFontValues, textBrushTimer, widthX75, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
					}
					else										// > 1 hour, then show hr:min:sec format + move text to right
					{
						timeLeft = barTimeLeft.Hours.ToString("00") + ":" + barTimeLeft.Minutes.ToString("00") + ":" + barTimeLeft.Seconds.ToString("00");
						graphics.DrawString(timeLeft, textFontValues, textBrushTimer, widthX100, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
					}
				}
				// if "DisplayTime is not true, then likely either outside of session hours or there is no data connection
				else
					graphics.DrawString("closed", textFontValues,textBrushNormal, widthX100, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
			}
			// For a RANGE BAR CHART, display the max Hi and Lo value that current bar can reach before a new bar is started
			// also display size in ticks for range of current bar. User input for number of ticks remaining to change color of text.
			else
			if (range > 0)
			{
				graphics.DrawString(rangeHi.ToString(decimals), textFontSmValues,textBrushHighs, widthX75, bounds.Y + (bounds.Height*location_Timer/100)-6, formatCenter);
				graphics.DrawString(rangeLo.ToString(decimals), textFontSmValues,textBrushLows, widthX75, bounds.Y + (bounds.Height*location_Timer/100)+14, formatCenter);
				if (remainingRange <= alertTimerLast)
					graphics.DrawString(remainingRange.ToString(), textFontValues,textBrushTimer0, widthX25, bounds.Y + (bounds.Height*location_Timer/100), formatCenter);
				else if (remainingRange <= alertTimer1st)
					graphics.DrawString(remainingRange.ToString(), textFontValues,textBrushTimer1, widthX25, bounds.Y + (bounds.Height*location_Timer/100), formatCenter);
				else
					graphics.DrawString(remainingRange.ToString(), textFontValues,textBrushNormal, widthX25, bounds.Y + (bounds.Height*location_Timer/100), formatCenter);
			}
			// For a TICK CHART, display ticks remaining. User input for number of ticks remaining to change color of text.
			else
			if (Bars.Period.Id == PeriodType.Tick)
			{
				if (remainingTicks <= alertTimerLast)
					graphics.DrawString(remainingTicks.ToString()+"t", textFontValues,textBrushTimer0, widthX75, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
				else if (remainingTicks <= alertTimer1st)
					graphics.DrawString(remainingTicks.ToString()+"t", textFontValues,textBrushTimer1, widthX75, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
				else
					graphics.DrawString(remainingTicks.ToString()+"t", textFontValues,textBrushNormal, widthX75, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
				if(remainingTicks <= alertTimerLast && audioTimerAlertPlay)
					SoundTimerAlert(remainingTicks+1);
			}
			// For a VOLUME CHART, display volume remaining. User input for volume remaining to change color of text.
			else
			if (Bars.Period.Id == PeriodType.Volume)
			{
				if (remainingVol <= alertTimerLast)
					graphics.DrawString(remainingVol.ToString()+"v", textFontValues,textBrushTimer0, widthX75, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
				else if (remainingVol <= alertTimer1st)
					graphics.DrawString(remainingVol.ToString()+"v", textFontValues,textBrushTimer1, widthX75, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
				else
					graphics.DrawString(remainingVol.ToString()+"v", textFontValues,textBrushNormal, widthX75, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
				if(remainingVol <= alertTimerLast && audioTimerAlertPlay)
					SoundTimerAlert(remainingVol+1);
			}
			// For non intra-day chart show error message
			else
			{
				graphics.DrawString("no timer", textFontValues,textBrushNormal, widthX100, bounds.Y + (bounds.Height*location_Timer/100), formatRight);
				if (timer != null)
					timer.Enabled = false;
			}

			// PERCENT COMPLETE display of bar progress (% complete).
			// User input to enable display - location is a fixed offset from position of countdown timer
			if (ShowTimerPercent)
			{
				// RangeBars not supported, so this is a work around
				if (range > 0)
					currentPercent = (int)(currentRange/range*100);
				else
					currentPercent	= (int)(Bars.PercentComplete*100);
				graphics.DrawString(currentPercent.ToString()+"%", textFontValues,textBrushNormal, widthX75, bounds.Y + (bounds.Height*location_Timer/100)-30, formatRight);
				// Substitute remainingPercent for currentPercent in DrawString method to get countdown
				// remainingPercent = 100 - currentPercent);
			}
			
			/////////////////////////////////////////
			// CCI POINTS DIFFERENCE AND CCI VALUE
			//
			// CCI points difference determine if ZLR entry is a go or no-go. >=15 points is required for the basic nuance.
			// This and other criteria are checked in OnBarUpdate and when all are met "zlrOK" or "zlrVG" status is set to change text color.
			// ***NOTE*** CZI, SideWinder and a hook (or reject of zero) are not included here.
			// User input to enable display and set vertical location.
			//
			if (showCCIpoints)
			{
				// current bar CCI difference
				graphics.DrawString(cciPointsDifference.ToString(), textFontValues, zlrVG ? textBrushCCIvg : zlrOK ? textBrushCCIok : textBrushCCInotOK, widthX75, bounds.Y + (bounds.Height*location_CCIpoints/100), formatCenter);
	//			graphics.DrawString(cciPointsDifference.ToString(), textFontValues, textBrushNormal, widthX75, bounds.Y + (bounds.Height*location_CCIpoints/100), formatCenter);
				// prior bar CCI difference			
				graphics.DrawString(cciPointsPriorDiff.ToString(), textFontValues,textBrushCCIprior, widthX25, bounds.Y + (bounds.Height*location_CCIpoints/100), formatCenter);
				// current bar CCI value now
				graphics.DrawString(cciValue.ToString("f2"), textFontSmValues,textBrushCCIprior, widthX75, bounds.Y + (bounds.Height*location_CCIvalue/100), formatCenter);
				// prior bar CCI value at close
				graphics.DrawString(cciValuePrior.ToString("f2"), textFontSmValues,textBrushCCIprior, widthX25, bounds.Y + (bounds.Height*location_CCIvalue/100), formatCenter);
			}
			
			/////////////////////////////////////////
			// Stamp with version information
			graphics.DrawString(".", textFontLabels,textBrushVersion, widthX, bounds.Y + bounds.Height - 10, stringFormat);
			
		} //end plot graphics
		
		
		/////////////////////////////////////////
		// Additional Minutes Timer Code
		private bool DisplayTime()
        {
			if (ChartControl != null
				&& Bars != null
				&& Bars.Count > 0
				&& Bars.MarketData != null
				&& Bars.MarketData.Connection.PriceStatus == Cbi.ConnectionStatus.Connected
				&& (Bars.MarketData.Connection.Options.Provider != Cbi.Provider.OpenTick || !(Bars.MarketData.Connection.Options as Cbi.OpenTickOptions).UseDelayedData)
				&& Bars.Session.InSession(Now, Bars.Period, true) )
				return true;

            return false;
        }

        public override void Dispose()
        {
            if (timer != null)
            {
                timer.Enabled = false;
                timer = null;
            }
            base.Dispose();
        }

		private DateTime Now
		{
			get { return (Bars.MarketData.Connection.Options.Provider == Cbi.Provider.Replay ? Bars.MarketData.Connection.Now : DateTime.Now); }
		}

		private void OnTimerTick(object sender, EventArgs e)
        {
			if(DisplayTime()) 
				ChartControl.ChartPanel.Invalidate();                
        }
		
		private void SoundTimerAlert(int rearm)
		{
			// Build a unique alert id - this will keep multiple charts displaying same instrument, period and alert time from
			// firing simultaneously
			string uniqueName = Bars.MarketData.Instrument.FullName.ToString() + Bars.Period.ToString()+alertTimerLast.ToString();
			
			// audioTimerAlertFile should be a .wav file placed in the Ninja install directory e.g. C:\Program Files\NinjaTrader 7\sounds
        	Alert(uniqueName, Priority.High, "Period Closing Alert", audioTimerAlertFile, rearm, Color.Gray, Color.Black);
		}
		
		#endregion

        #region Properties
        [Description("14 bar period is standard, 20 bar on daily charts.")]
        [Category("Parameters")]
        public int Period
        {
            get { return period; }
            set { period = Math.Max(1, value); }
        }
		
		[Description("Vertical location for Timer. Enter distance from top as percentage (0-99%)")]
        [Category("Panel")]
        public int Location_Timer
        {
            get { return location_Timer; }
            set { location_Timer = Math.Max(0, value); }
        }
		
		[Description("Display Timer as total seconds, otherwise show as Min:Sec")]
        [Category("Panel")]
        public bool ShowTimerAsSeconds
        {
            get { return showTimerAsSeconds; }
            set { showTimerAsSeconds = value; }
        }
		
        [Description("Show additional Timer giving bar progress as percentage completed")]
        [Category("Panel")]
        public bool ShowTimerPercent
        {
            get { return showTimerPercent; }
            set { showTimerPercent = value; }
        }
		
		[Description("Period remaining-1st Timer color change. Price increment(ticks) for Range; Seconds for Minute chart")]
        [Category("Panel")]
        public int AlertTimer1st
        {
            get { return alertTimer1st; }
            set { alertTimer1st = Math.Max(0, value); }
        }
		
		[Description("Period remaining-Last Timer color change. Price increment(ticks) for Range; Seconds for Minute chart")]
        [Category("Panel")]
        public int AlertTimerLast
        {
            get { return alertTimerLast; }
            set { alertTimerLast = Math.Max(0, value); }
        }

        [Description("Vertical location for Prices. Enter distance from top as percentage (0-99%)")]
        [Category("Panel")]
        public int Location_Price
        {
            get { return location_Price; }
            set { location_Price = Math.Max(0, value); }
        }

        [Description("Show close price of prior bar")]
        [Category("Panel")]
        public bool ShowLastClose
        {
            get { return showLastClose; }
            set { showLastClose = value; }
        }
		
		[Description("Show High/Low of Day (Not acurate on Range Bars with Gap Open)")]
        [Category("Panel")]
        public bool ShowDayHiLo
        {
            get { return showDayHiLo; }
            set { showDayHiLo = value; }
		}
		
		[Description("Vertical location for Day High/Low. Enter distance from top as percentage (0-99%)")]
        [Category("Panel")]
        public int Location_DayHiLo
        {
            get { return location_DayHiLo; }
            set { location_DayHiLo = Math.Max(0, value); }
        }
		
		[Description("Show EMA slope and distance")]
        [Category("Panel")]
        public bool ShowEMA
        {
            get { return showEMA; }
            set { showEMA = value; }
		}
		
		[Description("Vertical location for EMA. Enter distance from top as percentage (0-99%)")]
        [Category("Panel")]
        public int Location_EMA
        {
            get { return location_EMA; }
            set { location_EMA = Math.Max(0, value); }
        }
		
		[Description("Show LSMA slope and distance")]
        [Category("Panel")]
        public bool ShowLSMA
        {
            get { return showLSMA; }
            set { showLSMA = value; }
		}
		
		[Description("Vertical location for LSMA. Enter distance from top as percentage (0-99%)")]
        [Category("Panel")]
        public int Location_LSMA
        {
            get { return location_LSMA; }
            set { location_LSMA = Math.Max(0, value); }
        }
		
		[Description("Show CCI Points Difference between bars, and CCI value")]
        [Category("Panel")]
        public bool ShowCCIpoints
        {
            get { return showCCIpoints; }
            set { showCCIpoints = value; }
		}
		
		[Description("Vertical location for CCI points Difference. Enter distance from top as percentage (0-99%)")]
        [Category("Panel")]
        public int Location_CCIpoints
        {
            get { return location_CCIpoints; }
            set { location_CCIpoints = Math.Max(0, value); }
        }
		
		[Description("Vertical location for CCI Value. Enter distance from top as percentage (0-99%)")]
        [Category("Panel")]
        public int Location_CCIvalue
        {
            get { return location_CCIvalue; }
            set { location_CCIvalue = Math.Max(0, value); }
        }
		
		[Description("Show panel background fill color. Otherwise will use your chart background color")]
        [Category("Panel")]
        public bool ShowPanelColor
        {
            get { return showPanelColor; }
            set { showPanelColor = value; }
		}
		
		[Description("Price difference (in ticks) to change text color for HoD LoD Open")]
        [Category("Panel")]
        public int Threshold_DayHiLo
        {
            get { return threshold_DayHiLo; }
            set { threshold_DayHiLo = Math.Max(0, value); }
        }
		
		[Description("Audio Timer Alert file to be played")]
        [Category("Panel")]
        public string AudioTimerAlertFile
        {
            get { return audioTimerAlertFile; }
            set { audioTimerAlertFile = value; }
        }
		
		[Description("Play Audio Timer Alert")]
        [Category("Panel")]
        public bool AudioTimerAlertPlay
        {
            get { return audioTimerAlertPlay; }
            set { audioTimerAlertPlay = value; }
        }
		
		[Description("Left edge of Panel, larger number inceases width, change chart Right Side Margin slightly more")]
        [Category("Panel")]
        public int PanelLeftEdge
        {
            get { return panelLeftEdge; }
            set { panelLeftEdge = Math.Max(4, value); }
        }
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private CCIpanel[] cacheCCIpanel = null;

        private static CCIpanel checkCCIpanel = new CCIpanel();

        /// <summary>
        /// The CCI Panel w
        /// </summary>
        /// <returns></returns>
        public CCIpanel CCIpanel(int period)
        {
            return CCIpanel(Input, period);
        }

        /// <summary>
        /// The CCI Panel w
        /// </summary>
        /// <returns></returns>
        public CCIpanel CCIpanel(Data.IDataSeries input, int period)
        {
            if (cacheCCIpanel != null)
                for (int idx = 0; idx < cacheCCIpanel.Length; idx++)
                    if (cacheCCIpanel[idx].Period == period && cacheCCIpanel[idx].EqualsInput(input))
                        return cacheCCIpanel[idx];

            lock (checkCCIpanel)
            {
                checkCCIpanel.Period = period;
                period = checkCCIpanel.Period;

                if (cacheCCIpanel != null)
                    for (int idx = 0; idx < cacheCCIpanel.Length; idx++)
                        if (cacheCCIpanel[idx].Period == period && cacheCCIpanel[idx].EqualsInput(input))
                            return cacheCCIpanel[idx];

                CCIpanel indicator = new CCIpanel();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Period = period;
                Indicators.Add(indicator);
                indicator.SetUp();

                CCIpanel[] tmp = new CCIpanel[cacheCCIpanel == null ? 1 : cacheCCIpanel.Length + 1];
                if (cacheCCIpanel != null)
                    cacheCCIpanel.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheCCIpanel = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The CCI panel w
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.CCIpanel CCIpanel(int period)
        {
            return _indicator.CCIpanel(Input, period);
        }

        /// <summary>
        /// The CCI panel w
        /// </summary>
        /// <returns></returns>
        public Indicator.CCIpanel CCIpanel(Data.IDataSeries input, int period)
        {
            return _indicator.CCIpanel(input, period);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The CCI panel w
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.CCIpanel CCIpanel(int period)
        {
            return _indicator.CCIPanel(Input, period);
        }

        /// <summary>
        /// The CCI panel w
        /// </summary>
        /// <returns></returns>
        public Indicator.CCIPanel CCIPanel(Data.IDataSeries input, int period)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.CCIPanel(input, period);
        }
    }
}
#endregion
